(*^
::[	Information =

	"This is a Mathematica Notebook file.  It contains ASCII text, and can be
	transferred by email, ftp, or other text-file transfer utility.  It should
	be read or edited using a copy of Mathematica or MathReader.  If you 
	received this as email, use your mail application or copy/paste to save 
	everything from the line containing (*^ down to the line containing ^*)
	into a plain text file.  On some systems you may have to give the file a 
	name ending with ".ma" to allow Mathematica to recognize it as a Notebook.
	The line below identifies what version of Mathematica created this file,
	but it can be opened using any other version as well.";

	FrontEndVersion = "Macintosh Mathematica Notebook Front End Version 2.2";

	MacintoshStandardFontEncoding; 
	
	fontset = title, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e8,  24, "Times"; 
	fontset = subtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e6,  18, "Times"; 
	fontset = subsubtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, italic, e6,  14, "Times"; 
	fontset = section, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, grayBox, M22, bold, a20,  18, "Times"; 
	fontset = subsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, blackBox, M19, bold, a15,  14, "Times"; 
	fontset = subsubsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, whiteBox, M18, bold, a12,  12, "Times"; 
	fontset = text, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = smalltext, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = input, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeInput, M42, N23, bold, L-5,  10, "Courier"; 
	fontset = output, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = message, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, R65535, L-5,  12, "Courier"; 
	fontset = print, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = info, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, B65535, L-5,  12, "Courier"; 
	fontset = postscript, PostScript, formatAsPostScript, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeGraphics, M7, l34, w245, h249,  12, "Courier"; 
	fontset = name, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7, italic,  10, "Geneva"; 
	fontset = header, inactive, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = leftheader, inactive, L2,  12, "Times"; 
	fontset = footer, inactive, noKeepOnOnePage, preserveAspect, center, M7,  12, "Times"; 
	fontset = leftfooter, inactive, L2,  12, "Times"; 
	fontset = help, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = clipboard, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = completions, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special1, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special2, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special3, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special4, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special5, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	automaticGrouping; currentKernel; 
]
:[font = title; inactive; preserveAspect; startGroup]
Bras and Kets
:[font = subsubtitle; inactive; preserveAspect]
Colin P. Williams
:[font = section; inactive; locked; preserveAspect; startGroup]
Copyright Notice
:[font = text; inactive; locked; preserveAspect; endGroup]
Copyright Colin P. Williams (1997).

This Notebook is intended to be used in conjunction with "Explorations in Quantum Computing" by Colin P. Williams and Scott H. Clearwater, TELOS, Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to copy and distribute this Notebook freely for any non-commercial activity provided you include this copyright notice at the beginning of all such copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on "Explorations in Quantum Computing" check out the TELOS web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To order call 1-800-777-4643.

All other rights reserved.
:[font = section; inactive; locked; preserveAspect; startGroup]
How to use this Notebook
:[font = text; inactive; locked; preserveAspect]
This Notebook supplies utilities needed in the quantum computingsimulations. These include utilities for working with bras and kets, generating random (test) superpositions, extracting amplitudes and probabilities from a ket, normalizing kets etc.  Load this Notebook when you run Feynman's quantum computer.

The Notebook contains definitions for:
:[font = input; locked; preserveAspect]
KetToColumnVector
BraToRowVector
ColumnVectorToKet
RowVectorToBra
KetToBra
BraToKet
ExpectationValue
BasisEigenstates
Amplitudes
Probabilities
SymbolicCoefficients
NormalizedKetQ
NormalizeKet
Direct
TruthTable
HermitianQ
UnitaryQ
SymbolicSuperposition
RandomSuperposition
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Converting Kets to Column Vectors
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
KetToColumnVector[ket[0]]:={{1},{0}}      (* spin up   = 0 *)
KetToColumnVector[ket[1]]:={{0},{1}}      (* spin down = 1 *)
KetToColumnVector[ket[bits__]]:=
	Apply[Direct, Map[KetToColumnVector[ket[#]]&, {bits}]]

KetToColumnVector[a_ ket_ket]:=
	a KetToColumnVector[ket]
	
KetToColumnVector[Plus[ket_, kets___]]:=
	Apply[Plus, Map[KetToColumnVector, {ket, kets}]]
	
KetToColumnVector[superposition_]:=
	KetToColumnVector[ Expand[superposition] ] 
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Converting Bras to Row Vectors
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
BraToRowVector[bra[0]]:={{1,0}}
BraToRowVector[bra[1]]:={{0,1}}
BraToRowVector[w_. bra[bits__]]:=
	w * Apply[Direct, Map[BraToRowVector[bra[#]]&, {bits}]]
BraToRowVector[w_. bra[bits__] + bras_.]:=
	BraToRowVector[w * bra[bits]] + BraToRowVector[bras]
BraToRowVector[superposition_]:=
	BraToRowVector[Expand[superposition]]
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Converting Column Vectors to Kets
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
ColumnVectorToKet[amplitudes_]:=
	Apply[Plus,
		  MapThread[(#1[[1]] #2)&,
		            {amplitudes,
		             EigenKets[ Length[amplitudes] ]
		            }
		           ]
		 ]
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Converting Row Vectors To Bras
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
RowVectorToBra[{{wi__}}]:=
	Module[{eigenBras},
		eigenBras = EigenKets[Length[{wi}]] /. ket->bra;
		Apply[Plus, MapThread[(#1 #2)&, {{wi}, eigenBras}]]
	]
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Converting Between Bras and Kets
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
KetToBra[ket_]:=
	RowVectorToBra[Conjugate[Transpose[KetToColumnVector[ket]]]]

BraToKet[bra_]:=
	ColumnVectorToKet[Conjugate[Transpose[BraToRowVector[bra]]]]
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Average Value of an Observable
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
ExpectationValue[w_. ket[bits__] + kets_., observable_]:=
	(If[!HermitianQ[observable], 
		(Message[ExpectationValue::notHermitian]; Abort[]),
		If[Length[observable] != 2^Length[{bits}],
		   (Message[ExpectationValue::incompatible]; Abort[])]];
		       
	 (BraToRowVector[KetToBra[w * ket[bits] + kets]] . 
	  observable . 
	  KetToColumnVector[w * ket[bits] + kets]
	 )[[1,1]]  (* scalar = a 1 x 1 matrix, [[1,1]] removes the parentheses *)
	)

ExpectationValue[superposition_, observable_]:=
	ExpectationValue[Expand[superposition], observable]

ExpectationValue::notHermitian =
	"Your purported observable is not an Hermitian matrix.";
ExpectationValue::incompatible =
	"The dimensions of the state vector and observable are incompatible.";

(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Creating Eigenstates that Span a Hilbert Space
:[font = input; locked; initialization; preserveAspect]
*)
BasisEigenstates[m_Integer]:= EigenKets[2^m]

BasisEigenstates::usage = 
  "BasisEigenstates[m] returns the complete set of \
  eigenstates that span the Hilbert space of an m-bit \
  quantum memory register.";
(*
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
EigenKets[n_]:=
	Module[{bits},
		bits = Table[Apply[ket, IntegerDigits[i,2]], 
		             {i, 0, n-1}];
		          (* last eigenket has the most bits *)
		Map[PadTo[Length[Last[bits]], #]&, bits]
	]

PadTo[nDigits_, digits_]:=
	Join[Apply[ket, Table[0,{nDigits - Length[digits]}]], 
	     digits]
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Accessing Amplitudes of Superpositions and Computing Probabilities
:[font = input; locked; initialization; preserveAspect; startGroup]
*)
Options[Amplitudes] = {ShowEigenstates->False};

ShowEigenstates::usage = 
	"ShowEigenstates is an option for Amplitudes that \
	determines whether the 
output should be a list of the \
	amplitudes or a list of {eigenstate, 
amplitude} pairs.";

Amplitudes[w_. ket[bits__] + kets_., opts___]:=
	Module[{showeigen},
	showeigen = ShowEigenstates /. {opts} /. Options[Amplitudes];
	Which[showeigen == True, 
			Map[{#, Coefficient[w ket[bits] + kets, #]}&,
		        BasisEigenstates[ Length[{bits}] ]
		       ],
		  showeigen == False,
		    Map[Coefficient[w ket[bits] + kets, #]&,
		        BasisEigenstates[ Length[{bits}] ]
		       ]
		  ]
	]

(* This clause catches cases like 1/Sqrt[2] (ket[0] + ket[1]) etc *)	
Amplitudes[c_ (w_. ket[bits__] + kets_.)]:=
	Amplitudes[ Expand[c (w ket[bits] + kets)] ]

Amplitudes::usage = 
  "Amplitudes[superposition] returns the amplitudes of the \
  eigenstates in a superposition or ket vectors.";
(*
:[font = message; inactive; locked; preserveAspect; endGroup]
General::spell1: 
   Possible spelling error: new symbol name "Amplitudes"
     is similar to existing symbol "amplitudes".
:[font = input; locked; initialization; preserveAspect]
*)
Options[Probabilities] = {ShowEigenstates->False};

Probabilities[w_. ket[bits__] + kets_., opts___]:=
	Module[{showeigen, amplitudes, symbols, sumOfSquares},
		showeigen    = ShowEigenstates /. {opts} /. Options[Probabilities];
		amplitudes   = Amplitudes[w ket[bits] + kets];
		symbols      = SymbolicCoefficients[amplitudes]; (*see below*)
		sumOfSquares = Simplify[
		                Apply[Plus, 
		                      Map[ComplexExpand[Abs[#]^2, symbols]&, 
		                          amplitudes]]];
		amplitudes   = If[sumOfSquares=!=1,  (* renormalize amplitudes
		                                         if necessary *)
		                  amplitudes/Sqrt[sumOfSquares],
		                  amplitudes];
		Which[showeigen == True,  
		       MapThread[{#1, ComplexExpand[Abs[#2]^2, symbols]}&, 
		                 {BasisEigenstates[Length[{bits}]], amplitudes}
		                ],
			  showeigen == False, 
			   Map[ComplexExpand[Abs[#]^2, symbols]&, amplitudes]
	    ]
	]

Probabilities[c_ (w_. ket[bits__] + kets_.)]:=
	Probabilities[ Expand[c (w ket[bits] + kets)] ]

Probabilities::usage =
	"Probabilities[superposition] returns the probabilities of \
	 finding a system in a state described by superposition in \
	 each of its possible eigenstates upon being measured (observed). \
	 If Probabilities is given the option ShowEigenstates->True \
	 the function returns a list of {eigenstate, probability} pairs.";
(*
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
SymbolicCoefficients[amplitudes_List]:=
	Select[Union[Flatten[Map[Variables, amplitudes]]], 
		   Not[MatchQ[#, Abs[_]]]&]
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Testing Whether a Ket is Properly Normalized
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
Needs["Algebra`ReIm`"];

NormalizedKetQ[ket_]:=
	Module[{columnVector},
		columnVector = KetToColumnVector[ket];
		(Inner[Times, 
		       Conjugate[Transpose[columnVector]], 
               columnVector,
               Plus
              ] == {{1}} // N ) /. z_ Conjugate[z_] :> Abs[z]^2
    ]
   
NormalizedKetQ::usage =
	"NormalizedKetQ[ket] returns True if the square \
	moduli of the amplitudes of the eigenkets in the \
	superposition \"ket\" sum to 1. If \"ket\" has non-numeric \
	amplitudes, the normalization cannot always be determined.";
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
NormalizeKet
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
NormalizeKet[superposition_]:=
	superposition /; NormalizedKetQ[superposition]
NormalizeKet[superposition_]:=
	Expand[superposition / 
	       Sqrt[Apply[Plus, 
	                  Map[Abs[#]^2&, 
	                      Amplitudes[superposition, 
	                                 ShowEigenstates->False]
	                     ]
	                 ]
	           ]
	]
	      
NormalizeKet::usage =
	"NormalizeKet[superposition] is used to normalize a given \
	superposition of
 ket vectors. That is, if the sum of the squares \
	of the absolute values of 
the amplitudes of the eigenstates in \
	the superposition do not sum to 1, 
NormalizeKet rescales the \
	amplitudes so that they squared moduli will sum 
to 1.";
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Direct Product
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
(* Last modified 09/07/96 *)
Needs["LinearAlgebra`MatrixManipulation`"];

Direct[op1_, op2_]:=
	BlockMatrix[Outer[Times, op1, op2]] /; MatrixQ[op1] && MatrixQ[op2]
	
Direct[ket_, bra_]:=
	Direct[KetToColumnVector[ket], BraToRowVector[bra]] /; IsKetQ[ket] && IsBraQ[
bra]
	
Direct[ket1_, ket2_]:=
	ColumnVectorToKet[
		Direct[KetToColumnVector[ket1],
	           KetToColumnVector[ket2]]
	]/; IsKetQ[ket1] && IsKetQ[ket2]

Direct[bra1_, bra2_]:=
	RowVectorToBra[
		Direct[BraToRowVector[bra1],
			   BraToRowVector[bra2]]
	] /; IsBraQ[bra1] && IsBraQ[bra2]
	
Direct[bra_, ket_]:=
	(Message[Direct::braket];
	 Direct[BraToRowVector[bra], KetToColumnVector[ket]]) /; IsBraQ[bra] && 
IsKetQ[ket]

Direct[bra_, op_]:=
	(Message[Direct::braop];
	 Direct[BraToRowVector[bra], op]) /; IsBraQ[bra] && MatrixQ[op]
	
Direct[op_, bra_]:=
	(Message[Direct::opbra];
	 Direct[op, BraToRowVector[bra]]) /; MatrixQ[op] && IsBraQ[bra]
	
Direct[ket_, op_]:=
	(Message[Direct::ketop];
	 Direct[KetToColumnVector[ket], op]) /; IsKetQ[ket] && MatrixQ[op]
	
Direct[op_, ket_]:=
	(Message[Direct::opket];
	 Direct[op, KetToColumnVector[ket]]) /; MatrixQ[op] && IsKetQ[ket]

Direct[matrices__]:=
	Fold[Direct, First[{matrices}], Rest[{matrices}]]

Direct::braket =
	"Warning - You are taking the DIRECT product of a bra \
	and a ket. This is 
unusual. Perhaps you meant to use \
	the DOT product?";
	
Direct::braop =
	"Warning - You are taking the DIRECT product of a bra \
	with an operator. 
This is unusual. Perhaps you meant to use \
	the DOT product?";
	
Direct::opbra =
	"Warning - You are taking the DIRECT product of an operator \
	with a bra. 
This is unusual. Perhaps you meant to use \
	the DOT product?";

Direct::ketop =
	"Warning - You are taking the DIRECT product of a ket \
	with an operator. 
This is unusual. Perhaps you meant to use \
	the DOT product?";

Direct::opket =
	"Warning - You are taking the DIRECT product of an operator \
	with a ket. 
This is unusual. Perhaps you meant to use \
	the DOT product?";


IsKetQ[w_. ket[__] + kets_.]:= True
IsKetQ[_]:=False
	
IsBraQ[w_. bra[__] + bras_.]:= True
IsBraQ[_]:=False
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Truth Table of a Logic Gate
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
TruthTable[gate_]:=
	Module[{n,m},
		{n,m} = Dimensions[gate];
		Which[Not[n==m && IntegerQ[n] && IntegerQ[m]],
		      Message[TruthTable::notsquare]; Abort[],
		      Not[IntegerQ[Log[2, n]]],
		      Message[TruthTable::powerof2]; Abort[]
		     ];
		Map[(# -> ColumnVectorToKet[gate . KetToColumnVector[#]])&, 
		    EigenKets[n]
		   ]  // ColumnForm
	]
	
TruthTable::notsquare = 
  "Your input is not a square matrix and cannot, therefore, represent a \
  
reversible logic gate.";

TruthTable::powerof2 = 
  "Your input is not a matrix of dimensions (2^m) x (2^m) for integer m \
  
and cannot, therefore, represent a reversible logic gate that operates \
  on 
m bits.";
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Types of Operators (Matrices)
:[font = input; locked; preserveAspect]
HermitianQ[matrix_]:=
	matrix == Conjugate[Transpose[matrix]]
:[font = input; locked; initialization; preserveAspect; endGroup]
*)
UnitaryQ[matrix_]:=
  Module[{rows, cols},
	{rows, cols} = Dimensions[matrix];
	If[Not[IntegerQ[rows]] || 
	   Not[IntegerQ[cols]] || 
	   rows != cols, Message[UnitaryQ::notsquarematrix]];
	
	   Chop[Simplify[ComplexExpand[Conjugate[Transpose[matrix]]] - 
	                 ComplexExpand[Inverse[matrix]]
	                ]
	       ] == ZeroMatrix[rows, cols]
  ]

UnitaryQ::notsquarematrix =
  "Your input is not a square matrix.";
  
ZeroMatrix[rows_, cols_]:=
	Table[0, {rows}, {cols}]
(*
:[font = subsection; inactive; locked; Cclosed; preserveAspect; startGroup]
Tools for Making Test Superpositions
:[font = input; locked; initialization; preserveAspect]
*)
SymbolicSuperposition[m_]:=
	Apply[Plus,
	      MapThread[(#1 #2)&, 
	                {SymbolicAmplitudes[m], BasisEigenstates[m]}]
	]
	
SymbolicSuperposition::usage =
	"SymbolicSuperposition[m] creates a superposition of 2^m \
	eigenstates whose
 amplitudes are uninstantiated symbols. These \
	eigenstates represent the 
possible states of an m-bit memory \
	register of a quantum computer. This 
function is useful for \
	exploring the effects of quantum mechanical 
operations on \
	arbitrary superpositions. Note that the general form does not

	guarentee that the superposition is normalized.";
	
SymbolicAmplitudes[m_]:=
	(Clear[w];
	 Map[ToExpression["w"<>ToString[#]]&, Table[i,{i,0,2^m - 1}]]
	)
(*
:[font = input; locked; initialization; preserveAspect; startGroup]
*)
Options[RandomSuperposition] = {Normalized->True};

RandomSuperposition[m_, opts___]:=
	Module[{normalized},
		normalized = Normalized /. {opts} /. Options[RandomSuperposition];
		superposition = Apply[Plus,
	                          MapThread[(#1 #2)&, 
	                                    {RandomAmplitudes[m],
	                                     BasisEigenstates[m]}
	                                   ]
	                         ];
		Which[normalized==True, NormalizeKet[superposition],
	          normalized==False, superposition
	         ]
	]

RandomSuperposition::usage =
	"RandomSuperposition[m] creates a normalized superposition \
	of 2^m eigenstates whose amplitudes are random complex numbers. \
	These eigenstates represent the possible states of an m-bit \
	memory register of a quantum computer. You can generate an \
	un-normalized superposition by setting the option Normalized->False.";
	
(* You can pick the amplitudes according to whatever distribution
   you like. In the current case we pick random complex numbers
   uniformly from the square in the complex plane bounded by a lower
   left corner at (-1,-I) and an upper right corner at (1,I).
*)
RandomAmplitudes[m_]:=
	Table[Random[Complex, {-1-I, 1+I}], {2^m}]
(*
:[font = message; inactive; locked; preserveAspect; endGroup; endGroup; endGroup; endGroup]
General::spell1: 
   Possible spelling error: new symbol name "normalized"
     is similar to existing symbol "Normalized".
^*)
